<?php

namespace App\Helpers\Sudani;

use Exception;
use App\Helpers\Functions;
use Illuminate\Support\Facades\Log;
use App\Helpers\Sudani\SudaniStatus;

class SudaniAPI
{
    private $client;
    private $serviceUrl;
    private $providerKey;
    private $serviceCode;
    private $userName;
    private $userAgent = "http-client/1.0"; //Hakaweena-Client/1.0
    public function __construct()
    {
        $this->client = new \GuzzleHttp\Client();
        $this->serviceUrl = config('services.sudani.serviceUrl');
        $this->providerKey = config('services.sudani.provideKey');
        $this->userName = config('services.sudani.userName');
        $this->serviceCode = config('services.sudani.serviceCode');
    }
    public function initPay(string $phoneNumber): SudaniStatus
    {
        try {
            $token = $this->getToken();
            if ($token != null) {
                $mPhone = Functions::getPhone249($phoneNumber);
                $response = $this->client->request(
                    "POST",
                    $this->serviceUrl . 'InitPay/',
                    [
                        'headers' => [
                            'User-Agent' => $this->userAgent,
                            "Accept" =>  "application/json",
                            "Content-Type" =>  "application/json",
                            "token" => $token
                        ],
                        'json' => [
                            'msisdn' => $mPhone,
                            'serviceCode' => $this->serviceCode
                        ]
                    ]
                );
                if ($response->getStatusCode() == 200) {
                    $body = $response->getBody()->getContents();
                    $serviceResponse = json_decode($body);
                    Log::info("SpayService_initPay: response {$body}");
                    if (!$serviceResponse->status && $serviceResponse->responseCode == 106) {
                        return new SudaniStatus(SudaniStatus::PHONE_NUMBER_IN_CORRECT, $serviceResponse);
                    } else if (!$serviceResponse->status && $serviceResponse->responseCode == 121) {
                        return new SudaniStatus(SudaniStatus::USER_IS_ALREADY_SUBSCRIBE, $serviceResponse);
                    }
                    if ($serviceResponse->status && $serviceResponse->responseCode == 1) {
                        return new SudaniStatus(SudaniStatus::OK, $serviceResponse);
                    }
                    Log::critical("SpayService_InitPay: fail with status {$serviceResponse->responseCode}");
                    return new SudaniStatus(SudaniStatus::SUDANI_ERROR);
                } else {
                    Log::critical("SpayService_InitPay: Fail due to login fail");
                    return new SudaniStatus(SudaniStatus::SUDANI_LOGIN_FAILS);
                }
            } else {
                Log::critical("SpayService_InitPay: can't get the token.");
                return new SudaniStatus(SudaniStatus::FAILS);
            }
        } catch (Exception $e) {
            report($e);
            return new SudaniStatus(SudaniStatus::FAILS);
        }
    }

    public function payment(int $requestId, int $pin): SudaniStatus
    {
        try {
            $token = $this->getToken();
            if ($token != null) {
                $response = $this->client->request(
                    "POST",
                    $this->serviceUrl . 'Payment/',
                    [
                        'headers' => [
                            'User-Agent' => $this->userAgent,
                            "Accept" =>  "application/json",
                            "Content-Type" =>  "application/json",
                            "token" => $token
                        ],
                        'json' => [
                            'pin' => $pin,
                            'requestId' => $requestId
                        ]
                    ]
                );
                if ($response->getStatusCode() == 200) {
                    $body = $response->getBody()->getContents();
                    $serviceResponse = json_decode($body);
                    Log::info("SpayService_payment: response {$body}");
                    if (!$serviceResponse->status && $serviceResponse->responseCode == 105) {
                        return new SudaniStatus(SudaniStatus::PIN_CODE_INVALID, $serviceResponse);
                    } else if (!$serviceResponse->status && ($serviceResponse->responseCode == 121 || $serviceResponse->responseCode == 120 || $serviceResponse->responseCode == 500)) {
                        return new SudaniStatus(SudaniStatus::OK, $serviceResponse);
                    }
                    if ($serviceResponse->status && $serviceResponse->responseCode == 1) {
                        return new SudaniStatus(SudaniStatus::OK, $serviceResponse);
                    }
                    Log::critical("SpayService_Payment: fail with status {$serviceResponse->responseCode}");
                    return new SudaniStatus(SudaniStatus::SUDANI_ERROR);
                } else {
                    Log::critical("SpayService_Payment: Fail due to login fail");
                    return new SudaniStatus(SudaniStatus::SUDANI_LOGIN_FAILS);
                }
            } else {
                Log::critical("SpayService_Payment: can't get the token.");
                return new SudaniStatus(SudaniStatus::FAILS);
            }
        } catch (Exception $e) {
            report($e);
            return new SudaniStatus(SudaniStatus::FAILS);
        }
    }


    public function unSubscribe(string $phoneNumber): SudaniStatus
    {
        try {
            $token = $this->getToken();
            if ($token != null) {
                $mPhone = Functions::getPhone249($phoneNumber);
                $response = $this->client->request(
                    "POST",
                    $this->serviceUrl . 'UnSubscribe/',
                    [
                        'headers' => [
                            'User-Agent' => $this->userAgent,
                            "Accept" =>  "application/json",
                            "Content-Type" =>  "application/json",
                            "token" => $token
                        ],
                        'json' => [
                            'msisdn' => $mPhone,
                            'serviceCode' => $this->serviceCode
                        ]
                    ]
                );
                if ($response->getStatusCode() == 200) {
                    $body = $response->getBody()->getContents();
                    $serviceResponse = json_decode($body);
                    Log::info("SpayService_unSubscribe: response {$body}");
                    if (!$serviceResponse->status && $serviceResponse->responseCode == 106) {
                        return new SudaniStatus(SudaniStatus::PHONE_NUMBER_IN_CORRECT, $serviceResponse);
                    } else if (!$serviceResponse->status && $serviceResponse->responseCode == 121) {
                        return new SudaniStatus(SudaniStatus::USER_IS_ALREADY_SUBSCRIBE, $serviceResponse);
                    } else if (!$serviceResponse->status && $serviceResponse->responseCode == 116) {
                        return new SudaniStatus(SudaniStatus::OK, $serviceResponse);
                    }
                    if ($serviceResponse->status && $serviceResponse->responseCode == 1) {
                        return new SudaniStatus(SudaniStatus::OK, $serviceResponse);
                    }
                    Log::critical("SpayService_unSubscribe: fail with status {$serviceResponse->responseCode}");
                    return new SudaniStatus(SudaniStatus::SUDANI_ERROR);
                } else {
                    Log::critical("SpayService_unSubscribe: Fail due to login fail");
                    return new SudaniStatus(SudaniStatus::SUDANI_LOGIN_FAILS);
                }
            } else {
                Log::critical("SpayService_unSubscribe: can't get the token.");
                return new SudaniStatus(SudaniStatus::FAILS);
            }
        } catch (Exception $e) {
            report($e);
            return new SudaniStatus(SudaniStatus::FAILS);
        }
    }

    private function getToken(): ?string
    {
        $pubKey = $this->getPublicKey();
        if ($pubKey != null) {
            $pwd = config('services.sudani.password');
            $password = $this->getPassword($pubKey, $pwd);
            if ($password != null) {
                $token = $this->login($password);
                return $token;
            }
        }
        return null;
    }
    private function getPassword(string $pubKey, string $password): ?string
    {
        try {
            // $decodedPubKey = base64_decode($pubKey);
            // #dd($pubKey);
            // if ($decodedPubKey) {
            //     $rsa = new RSA();

            //     $bR = $rsa->loadKey(array(
            //         'e' => new BigInteger(65537),
            //         'n' => new BigInteger(substr($decodedPubKey, 0, -4), -256)
            //     ));
            //     $encrypted_data = $rsa->encrypt($password);
            //     $decodedPassword = base64_encode($encrypted_data);
            //     //$openss = openssl_public_encrypt($password, $encrypted_data, $decodedPubKey);
            //     return $decodedPassword;
            // }

            $token_url = config('services.sudani.tokenServiceUrl');
            $response = $this->client->request(
                "POST",
                $token_url . 'token-service',
                [
                    'headers' => [
                        'User-Agent' => $this->userAgent,
                        "Accept" =>  "application/json",
                        "Content-Type" =>  "application/json"
                    ],
                    'json' => [
                        'publicKey' => $pubKey,
                        'password' => $password
                    ]
                ]
            );
            if ($response->getStatusCode() == 200) {
                $a = json_decode($response->getBody()->getContents());
                return $a->password;
            }
            return null;
        } catch (\Exception $e) {
            report($e);
            return null;
        }
    }


    private function getPublicKey(): ?string
    {
        try {
            $response = $this->client->request(
                "POST",
                $this->serviceUrl . 'GetPublicKey/',
                [
                    'headers' => [
                        'User-Agent' => $this->userAgent,
                        "Accept" =>  "application/json",
                        "Content-Type" =>  "application/json"
                    ],
                    'json' => [
                        'providerKey' => $this->providerKey
                    ]
                ]
            );
            if ($response->getStatusCode() == 200) {
                $result = json_decode($response->getBody()->getContents());
                if ($result->status) {
                    return $result->publicKey;
                }
            }
            return null;
        } catch (\Exception $e) {
            report($e);
            return null;
        }
    }


    private function login(string $password): ?string
    {
        try {
            $response = $this->client->request(
                "POST",
                $this->serviceUrl . 'Login/',
                [
                    'headers' => [
                        'User-Agent' => $this->userAgent,
                        "Accept" =>  "application/json",
                        "Content-Type" =>  "application/json"
                    ],
                    'json' => [
                        'login' => $this->userName,
                        'password' => $password
                    ]
                ]
            );
            if ($response->getStatusCode() == 200) {
                $result = json_decode($response->getBody()->getContents());
                if ($result->status) {
                    return $result->token;
                }
            }
            return null;
        } catch (\Exception $e) {
            report($e);
            return null;
        }
    }
}
